//******************************************************************************
//  MSP430Fx20x1 Demo - bq26150 Authentication using CRC via HDQ 1-wire Protocol
//
//  The MSP430 performs the CRC calculations to authenticate the bq26150.
//  The main() function sets up the bq26150 to perform its own CRC calculation,
//  then the resulting CRC value is read from the bq26150.  This CRC result   
//  is then compared with the CRC calculated by the MSP430.  If the 32-bit
//  CRC values match, then output STATUS GOOD, otherwise STATUS FAIL.
//  A logic level HIGH on the CE input is required to start authentication.
//  ACLK = n/a, MCLK = calibrated DCO = 8 MHz, SMCLK = (MCLK / 4) = 2 MHz
//
//                MSP430F20x1
//             -----------------
//         /|\|              XIN|-
//          | |                 |
//          --|RST          XOUT|-
//            |                 |
//            |             P1.0|--> STATUS
//            |         P1.1/TA0|<-> HDQ
//
//  R. Wu
//  Texas Instruments Inc.
//  February 2010
//  Built with IAR Embedded Workbench Version: 4.21A
//******************************************************************************

#include "bq26150.h"
#include "..\..\lib\VLO_RAND\vlo_rand.h"
#include "..\..\lib\HDQ\HDQ.h"

#define P1_STATUS BIT0
#define STATUS_FAIL (P1OUT&=~P1_STATUS)
#define STATUS_GOOD (P1OUT|=P1_STATUS)
#define STATUS_TOGGLE (P1OUT^=P1_STATUS)

struct MSP430_Bytes4 {
  unsigned char byte0;
  unsigned char byte1;
  unsigned char byte2;
  unsigned char byte3;
};

struct MSP430_Words2 {
  unsigned int word0;
  unsigned int word1;
};

typedef union MSP430_Long1 {
  unsigned long        all;
  struct MSP430_Words2 word;
  struct MSP430_Bytes4 byte;
} MSP430_Long1;


void reset(void)
{
  WDTCTL = 0x00;                            // Psswd access violation will reset  
}

unsigned short CRC16(unsigned char Data, unsigned short Coeff,
                     unsigned short CrcWord)
{
  unsigned int i;
  unsigned char temp;
  unsigned char LsbChk;

  temp = Data;
  for (i = 0; i < 8; i++)
  {
    Data = (Data ^ CrcWord);
    CrcWord = (CrcWord >> 1);
    temp = (temp >> 1);
    LsbChk = (Data & BIT0);
    if (LsbChk == 1)
      CrcWord = (CrcWord ^ Coeff);
    Data = temp;
  }

  return CrcWord;
}

void main(void)
{
  unsigned char Coeff[POLYCOEFFNUMBYTES];
  unsigned char Seed[POLYSEEDNUMBYTES];
  unsigned char Random[RANDOMCHALLENGENUMBYTES];
  unsigned char DeviceKey[DEVICEKEYNUMBYTES];
  unsigned char ReadCRC[CRCNUMBYTES];
  MSP430_Long1  CoeffCRC;
  MSP430_Long1  CalculatedCRC;
  unsigned int  i;
  unsigned int  ctrl;

  WDTCTL = (WDTPW | WDTHOLD);               // Stop watchdog timer
  P1SEL = 0x00;                             // P1.x default as GPIO
  P1DIR = 0xFF;                             // P1.x default as outputs
  P1OUT = 0x00;                             // P1.x default logic LOW
  P1SEL &= ~P1_STATUS;                      // P1.x GPIO function
  P1DIR |= P1_STATUS;                       // P1.x STATUS output direction
  STATUS_FAIL;                              // Initialize STATUS = FAIL
  P2SEL = 0x00;                             // P2.x default as GPIO
  P2DIR = 0xFF;                             // P2.x default as outputs
  P2OUT = 0x00;                             // P2.x default logic LOW
  DCOCTL = CALDCO_8MHZ;                     // Set DCO for 8MHz using
  BCSCTL1 = CALBC1_8MHZ;                    // calibration data in Info Flash
  BCSCTL2 = DIVS_2;                         // SMCLK = (DCO / 4)

  HDQSetup();                               // Do basic HDQ setup
  
  // Read Encrypted Device ID registers & decrypt to get plaintext values
  i = DEVICEKEYNUMBYTES;
  DeviceKey[--i] = (HDQRead(EDK_11) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] = (HDQRead(EDK_10) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_09) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_08) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_07) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_06) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_05) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_04) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_03) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_02) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_01) ^ BQ26150CRYPTOKEYBYTE);
  DeviceKey[--i] =  (HDQRead(EDK_00) ^ BQ26150CRYPTOKEYBYTE);

  // Read Encrypted Polynomial Coefficients registers & decrypt to get plaintext
  i = POLYCOEFFNUMBYTES;
  Coeff[--i] = (HDQRead(EDP_HB) ^ BQ26150CRYPTOKEYBYTE);
  Coeff[--i] = (HDQRead(EDP_LB) ^ BQ26150CRYPTOKEYBYTE);

  // Read Encrypted Polynomial Seed registers & decrypt to get plaintext values
  i = POLYSEEDNUMBYTES;
  Seed[--i] = (HDQRead(EDS_HB) ^ BQ26150CRYPTOKEYBYTE);
  Seed[--i] = (HDQRead(EDS_LB) ^ BQ26150CRYPTOKEYBYTE);

  // Create Random Challenge using the known instability of the VLO
  i = RANDOMCHALLENGENUMBYTES;
  Random[--i] = (TI_getRandomIntegerFromVLO() % 256);
  Random[--i] = (TI_getRandomIntegerFromVLO() % 256);
  Random[--i] = (TI_getRandomIntegerFromVLO() % 256);
  Random[--i] = (TI_getRandomIntegerFromVLO() % 256);

  // Write to the Random Challenge registers
  HDQWrite(RC_03, Random[3]);
  if (HDQRead(RC_03) != Random[3])
  {
    reset();
  }
  HDQWrite(RC_02, Random[2]);
  if (HDQRead(RC_02) != Random[2])
  {
    reset();
  }
  HDQWrite(RC_01, Random[1]);
  if (HDQRead(RC_01) != Random[1])
  {
    reset();
  }
  HDQWrite(RC_00, Random[0]);
  if (HDQRead(RC_00) != Random[0])
  {
    reset();
  }

  // Set the AUTH bit in the CTRL register to start authentication
  HDQWrite(CTRL, AUTH);
    
  // Wait for the DONE bit in CTRL register to get set
  // If the bq26150 times out, need to reset myself
  while (1)
  {
    ctrl = HDQRead(CTRL);                   // Get CTRL register contents
    if ( ((ctrl & DONE) == DONE) && (ctrl != HDQ_ERROR) )
    {
      break;                                // Get out of loop & continue
    }
    if (ctrl == HDQ_ERROR)                  // Check if read request timed out
      reset();
  }

  // Read CRC value from bq26150
  ReadCRC[1] = HDQRead(CRC_HB);
  ReadCRC[0] = HDQRead(CRC_LB);
 
 // Initialize the Seed & Coeff values for the CRC calculator
  CalculatedCRC.byte.byte1 = Seed[1];
  CalculatedCRC.byte.byte0 = Seed[0];
  CoeffCRC.byte.byte1 = Coeff[1];
  CoeffCRC.byte.byte0 = Coeff[0];
  
  // HOST CRC: Put Random Challenge through the CRC calculator
  for (i = 0; i < RANDOMCHALLENGENUMBYTES; i++)
  {
    CalculatedCRC.word.word0 = CRC16(Random[i], CoeffCRC.word.word0,
                                     CalculatedCRC.word.word0);
  }
  // HOST CRC: Put Plaintext Device ID through the CRC calculator
  for (i = 0; i < DEVICEKEYNUMBYTES; i++)
  {
    CalculatedCRC.word.word0 = CRC16(DeviceKey[i], CoeffCRC.word.word0,
                                     CalculatedCRC.word.word0);
  }
  
  // Compare both bytes of the CRC read from bq26150 with my HOST CRC
  if ( (ReadCRC[1] == CalculatedCRC.byte.byte1) &&
       (ReadCRC[0] == CalculatedCRC.byte.byte0) )
  {
    STATUS_GOOD;                            // Output STATUS GOOD signal
  }

}
